﻿# Basic PyU3D example of how to create and modify polygon primitives.
# Written by MysteriXYZ.

# Declare constants
SCREEN_WIDTH = 800
SCREEN_HEIGHT = 600
SPEED_LIMIT = 200

# Import the PyU3D package
import PyU3D as u3d
from PyU3D._local import *
import PyU3D.host as host
import PyU3D.custom as custom

# Psyco is strongly recommended to boost the framerate
import psyco
psyco.full()



# derive a custom class for regular polygons from PyU3D.Polygon
class RegularPoly(u3d.Polygon):

  """
  Regular polygon in the XZ-plane.

  """

  def __init__(self, radius, *args, **kwargs):

    # inherit from the PyU3D.Polygon baseclass
    super(RegularPoly, self).__init__(*args, **kwargs)

    self.radius = radius
    self.setVertices()


  def setVertices(self):

    """
    Shape the regular polygon by setting the properties of its vertices.

    """

    verts = self.getVertices()
    angle = math.radians(360.0 / len(verts))
    for i, vert in enumerate(verts):
      cos = math.cos(angle * i)
      sin = math.sin(angle * i)
      x = cos * self.radius
      z = sin * self.radius
      vert.setPosition(x, 0.0, z)
      u = .5 + cos / 2.
      v = .5 - sin / 2.
      vert.setTexCoords(u, v)


  def addVertex(self):

    """
    Add a new vertex to this regular polygon.

    """

    # insert a vertex before the first vertex
    self.insertVertex()
    self.setVertices()


  def removeVertex(self):

    """
    Remove a vertex from this regular polygon.

    """

    # delete the first vertex
    self.deleteVertex(self.getVertices()[0])
    self.setVertices()



# Set U3D options

u3d.setLog("log.txt")
u3d.setZBufferFormat(32)
u3d.setMultiSampleType(0)


# initialize U3D and its host
host.init(
          SCREEN_WIDTH, SCREEN_HEIGHT, SPEED_LIMIT,
          "PyU3D polygon example",
          fullscreen=False
         )


# create an object to handle keyboard input
keyb = host.Keyboard()

# create an object to handle the mouse
mouse = host.Mouse()

# hide the cursor
mouse.setVisibility(False)

# should problems arise due to the cursor being hidden, try to use a transparent
# cursor instead of hiding it; comment out the line above and de-comment the
# line below
##mouse.setCursor(None)



# Create the scene objects

camera = u3d.Camera(SCREEN_WIDTH, SCREEN_HEIGHT)

light = u3d.DirectionalLight()


# initialize a regular polygon object as a hexagon
poly = RegularPoly(
                    radius=30.0,
                    vertex_count=6,
                    y=85.0,
                    texture=Texture("Gfx/conc02.jpg")
                  )


# set a font
arial12 = Font("Arial", 12, BOLDITALIC)



def main():

  # main loop
  while True:

    # process the current step of the game
    host.step()

    if keyb.keyIsPressed("esc"):
      host.exit()
    elif keyb.keyIsPressed("ins"):
      # add a vertex to the regular poly if the <Insert> key was pressed
      poly.addVertex()
    elif keyb.keyIsPressed("del"):
      # remove a vertex from the regular poly if the <Delete> key was pressed
      poly.removeVertex()

    # update the transformation and projection of the camera
    camera.update()

    # Display important statistics

    arial12.draw(10, 10, "fps: "+str(host.getFPS()))
    arial12.draw(10, 40, "triangles: "+str(getDrawnTriangleCount()))
    arial12.draw(10, 70, "draw calls: "+str(u3d.getDrawCallCount()))

    # Display control info

    arial12.draw(10, SCREEN_HEIGHT-80, "Press <Insert> key to add a vertex.")
    arial12.draw(10, SCREEN_HEIGHT-50, "Press <Delete> key to remove a vertex.")


if __name__ == '__main__':
  main()
